#!/usr/bin/perl
#
#   ora_dbinfo.pl - collect information about an Oracle database instance
#
# ---------------------------------------------------------
# Copyright 2011 - 2013, roveda
#
# This file is part of ORACLE_TOOLS.
#
# ORACLE_TOOLS is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# ORACLE_TOOLS is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ORACLE_TOOLS.  If not, see <http://www.gnu.org/licenses/>.
#
#
# ---------------------------------------------------------
# Synopsis:
#   perl ora_dbinfo.pl <configuration file>
#
# ---------------------------------------------------------
# Description:
#   This script gathers a lot of information about an Oracle database
#   instance and builds a text report, which is sent to the ULS.
#   A csv file with the major information about Oracle's version, 
#   installed components and used features is sent as an inventory
#   file.
#
#   Send any hints, wishes or bug reports to: 
#     roveda at universal-logging-system.org
#
# ---------------------------------------------------------
# Options:
#   See the configuration file.
#
# ---------------------------------------------------------
# Restrictions:
#   Information about Oracle RAC is currently not implemented.
#
# ---------------------------------------------------------
# Dependencies:
#   Misc.pm
#   Uls2.pm
#   uls-client-2.0-1 or later
#   You must set the necessary Oracle environment variables
#   or configuration file variables before starting this script.
#
# ---------------------------------------------------------
# Disclaimer:
#   The script has been tested and appears to work as intended,
#   but there is no guarantee that it behaves as YOU expect.
#   You should always run new scripts on a test instance initially.
#
# ---------------------------------------------------------
# Versions:
#
# "@(#)  ora_dbinfo.pl   0.01   2010-12-22   roveda"
# "@(#)  ora_dbinfo.pl   0.02   2011-01-04   roveda"
#        Added AS_VALUE.
# "@(#)  ora_dbinfo.pl   0.03   2011-11-11   roveda"
#        Added the GPL. Added the DBID. Added the list of installed options.
#        Database Links are probably vulnerable!?
# "@(#)  ora_dbinfo.pl   0.04   2012-09-07   roveda"
#        Added an inventory with version, instance name and components 
#        in csv format (inventory-oracle-<hostname>.csv)
# "@(#)  ora_dbinfo.pl   0.05   2013-01-20   roveda"
#        Added ORACLEOPTIONUSAGE and ORACLEPATCH to inventory output.
# "@(#)  ora_dbinfo.pl   0.06   2013-04-27   roveda"
#        Got some problems with long strings selected from v$parameter, 
#        now using rtrim() and linesize 32000, that works correctly.
#        Added 'is default' to parameter output.
# "@(#)  ora_dbinfo.pl   0.07   2013-06-23   roveda"
#        Added ENCRYPTED for tablespace information.
# "@(#)  ora_dbinfo.pl   0.08   2013-08-17   roveda"
#        Modifications to match the new single configuration file.
#        Added compression information for tablespaces.
# "@(#)  ora_dbinfo.pl   0.09   2013-09-28   roveda"
#        Added nls_instance_parameters.
#
#
#        Change also $VERSION later in this script!
#
# ===================================================================


use strict;
use warnings;
use File::Basename;
use File::Copy;

# These are my modules:
use lib ".";
use Misc 0.32;
use Uls2 1.10;

my $VERSION = 0.09;

# ===================================================================
# The "global" variables
# ===================================================================

# Name of this script.
my $CURRPROG = "";

# The default command to execute sql commands.
my $SQLPLUS_COMMAND = 'sqlplus -S "/ as sysdba"';

my $WORKFILEPREFIX;
my $TMPOUT1;
my $TMPOUT2;
my $LOCKFILE;
my $DELIM = "!";

# This hash keeps the command line arguments
my %CMDARGS;
# This keeps the contents of the configuration file
my %CFG;

# This keeps the settings for the ULS
my %ULS;

# That is used to give the workfiles a timestamp.
# If it has changed since the last run of this script, it
# will build new workfiles (e.g. when the system is rebooted).
# (similar to LAST_ONSTAT_Z for Informix)
my $WORKFILE_TIMESTAMP = "";

# This is to indicate "not available":
my $NA = "n/a";

# Use this to test for (nearly) zero:
my $VERY_SMALL = 1E-60;

# The $MSG will contain still the "OK", when reaching the end
# of the script. If any errors occur (which the script is testing for)
# the $MSG will contain "ERROR" or a complete error message, additionally,
# the script will send any error messages to the uls directly.
# <hostname> - "Oracle Database Server [xxxx]" - __watch_oracle9.pl - message
my $MSG = "OK";

# Holds the __$CURRPROG or $CFG{"IDENTIFIER"} just for easy usage.
my $IDENTIFIER;

# This hash keeps the documentation for the teststeps.
my %TESTSTEP_DOC;

# Keeps the version of the oracle software
my $ORACLE_VERSION = "";


# This keeps the complete report (as text)
my $REPORT = "";


# That keeps the inventory data
my $INVENTORY = "";

# The hostname where this script runs on
my $MY_HOST = "";



# ===================================================================
# The subroutines
# ===================================================================

sub output_error_message {
  # output_error_message(<message>)
  #
  # Send the given message(s), set the $MSG variable and
  # print out the message.

  $MSG = "ERROR";
  foreach my $msg (@_) { print STDERR "$msg\n" }
  foreach my $msg (@_) { uls_value($IDENTIFIER, "message", $msg, " ") }

} # output_error_message


# ------------------------------------------------------------
sub send_doc {
  # send_doc(<title> [, <as title>])
  #
  # If the <title> is found in the $TESTSTEP_DOC hash, then
  # the associated text is sent as documentation to the ULS.
  # Remember: the teststep must exist in the ULS before any
  #           documentation can be saved for it.
  # If the alias <as title> is given, the associated text is
  # sent to the ULS for teststep <as title>. So you may even
  # document variable teststeps with constant texts. You may
  # substitute parts of the contents of the hash value, before
  # it is sent to the ULS.

  my $title = $_[0];
  my $astitle = $_[1] || $title;

  if (%TESTSTEP_DOC) {
    if ($TESTSTEP_DOC{$title}) {
      # TODO: You may want to substitute <title> with <astitle> in the text?
      uls_doc($astitle, $TESTSTEP_DOC{$title})
    } else {
      print "No documentation for '$title' found.\n";
    }
  }

} # send_doc


# ------------------------------------------------------------
sub errors_in_file {
  # errors_in_file <filename>
  #
  # Check contents of e.g. $TMPOUT1 for ORA- errors.

  my $filename = $_[0];

  if (! open(INFILE, "<$filename")) {
    output_error_message(sub_name() . ": Error: Cannot open '$filename' for reading. $!");
    return(1);
  }

  my $L;

  while ($L = <INFILE>) {
    chomp($L);
    if ($L =~ /ORA-/i) {
      # yes, there have been errors.
      output_error_message(sub_name() . ": Error: There have been error(s) in file '$filename'!");
      return(1);
    }

  } # while

  if (! close(INFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$filename'. $!");
    return(1);
  }
  return(0); # everything ok
} # errors_in_file


# ------------------------------------------------------------
sub reformat_spool_file {
  # reformat_spool_file(<filename>)
  #
  # Reformats the spool file, removes unnecessary blanks surrounding
  # the delimiter, like this:
  #
  # ARTUS                         !          2097152000!            519569408
  # SYSTEM                        !          2097152000!            174129152
  # UNDOTS                        !          1048576000!             10027008
  #
  # ARTUS!2097152000!519569408
  # SYSTEM!2097152000!174129152
  # UNDOTS!1048576000!10027008
  #
  # This is necessary, because matching of constant expressions (like 'ARTUS')
  # would fail (the proper expression would be: 'ARTUS                         ').

  my $filename = $_[0];
  my $tmp_filename = "$filename.tmp";

  if (! open(INFILE, $filename)) {
    output_error_message(sub_name() . ": Error: Cannot open '$filename' for reading. $!");
    return(0);
  }

  if (! open(OUTFILE, ">$tmp_filename")) {
    output_error_message(sub_name() . ": Error: Cannot open '$tmp_filename' for writing. $!");
    return(0);
  }

  my $L;

  while($L = <INFILE>) {
    chomp($L);
    my @e = split($DELIM, $L);
    my $E;
    foreach $E(@e) {
      print OUTFILE trim($E), $DELIM;
    }
    print OUTFILE "\n";
  }

  if (! close(INFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$filename'. $!");
    return(0);
  }

  if (! close(OUTFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$tmp_filename'. $!");
    return(0);
  }

  if (! copy($tmp_filename, $filename)) {
    output_error_message(sub_name() . ": Error: Cannot copy '$tmp_filename' to '$filename'. $!");
    return(0);
  }

  if (! unlink($tmp_filename)) {
    output_error_message(sub_name() . ": Error: Cannot remove '$tmp_filename'. $!");
    return(0);
  }
} # reformat_spool_file



# ------------------------------------------------------------
sub print_file {
  my $f = $_[0];

  if (open(F, $f)) {
    while(my $Line = <F>) {
      chomp($Line);
      print "[" . $Line . "]\n";
    }
  }
  close(F);

} # print_file



# ------------------------------------------------------------
sub exec_sql {
  # exec_sql(<sql command> [, <print the spool file>]);
  #
  # Just executes the given sql statement against the current database instance.
  # If <print the spool file> is a true expression (e.g. a 1) the spool file
  # will be printed to stdout.

  # connect / as sysdba

  # Set nls_territory='AMERICA' to get decimal points.

  my $sql = "
    set echo off
    alter session set nls_territory='AMERICA';
    set newpage 0
    set space 0
    set linesize 32000
    set pagesize 0
    set feedback off
    set heading off
    set markup html off spool off

    set trimout on;
    set trimspool on;
    set serveroutput off;
    set define off;
    set flush off;

    set numwidth 20
    set colsep '$DELIM'

    spool $TMPOUT1;

    $_[0]

    spool off;
  ";

  print "\nexec_sql()\n";
  print "SQL: $sql\n";

  if (! open(CMDOUT, "| $SQLPLUS_COMMAND")) {
    output_error_message(sub_name() . ": Error: Cannot open pipe to '$SQLPLUS_COMMAND'. $!");
    return(0);   # error
  }
  print CMDOUT "$sql\n";
  if (! close(CMDOUT)) {
    output_error_message(sub_name() . ": Error: Cannot close pipe to sqlplus. $!");
    return(0);
  }

  # -----
  # Print the spool output if a true second parameter is given

  if ($_[1]) {
    print "-----[ $TMPOUT1 ]-----\n";
    print_file($TMPOUT1);
    print "----------------------\n";
  }

  reformat_spool_file($TMPOUT1);

  return(1);   # ok
} # exec_sql


# -------------------------------------------------------------------
sub do_sql {
  # do_sql(<sql>)
  #
  # Returns 0, when errors have occurred,
  # and outputs an error message,
  # returns 1, when no errors have occurred.

  # if (exec_sql($_[0])) {
  if (exec_sql(@_)) {
    if (errors_in_file($TMPOUT1)) {
      output_error_message(sub_name() . ": Error: there have been errors when executing the sql statement.");
      uls_send_file_contents($IDENTIFIER, "message", $TMPOUT1);
      return(0);
    }
    # Ok
    return(1);
  }

  output_error_message(sub_name() . ": Error: Cannot execute sql statement.");
  uls_send_file_contents($IDENTIFIER, "message", $TMPOUT1);

  return(0);

} # do_sql



# -------------------------------------------------------------------
sub clean_up {
  # clean_up()
  #
  # Remove all left over files at script end.

  title("Cleaning up");

  my @files = @_;

  foreach my $f (@files) {
    # Remove the file.
    if (-e $f) {
      print "Removing file '$f' ...";
      if (unlink($f)) {print "Done.\n"}
      else {print "Failed.\n"}
    }
  }

} # clean_up


# -------------------------------------------------------------------
sub send_runtime {
  # The runtime of this script
  # send_runtime(<start_secs> [, {"s"|"m"|"h"}]);

  # Current time minus start time.
  my $rt = time - $_[0];

  my $unit = uc($_[1]) || "S";

  if    ($unit eq "M") { uls_value($IDENTIFIER, "runtime", pround($rt / 60.0, -1), "min") }
  elsif ($unit eq "H") { uls_value($IDENTIFIER, "runtime", pround($rt / 60.0 / 60.0, -2), "h") }
  else                 { uls_value($IDENTIFIER, "runtime", pround($rt, 0), "s") }


} # send_runtime



# -------------------------------------------------------------------
sub append2report {
  # append2report(<separator>, <title>, <text>);
  #
  # The first character of the <separator> is taken 
  # to insert a line of 70 of these characters.
  # Append a title and the text to the report.
  # The <title> is turned to upper case, 
  # the <title> is followed by a blank line.
  # the <text> is appended as is, it should be 
  # formatted e.g. using make_text_report().
  #
  # =======================================...
  # <title>
  # 
  # <description>
  # <text>

  my ($separator, $title, $text) = @_;

  if ($separator) {
    if ($REPORT) { $REPORT .= "\n\n" }
    my $t = substr($separator x 70, 0, 70);
    $REPORT .= $t;
  }

  if ($title) {

    $title = uc($title);

    if ($REPORT) { $REPORT .= "\n" }
    $REPORT .= $title;
    if ($REPORT) { $REPORT .= "\n" }
  }

  if ($text) {
    if ($REPORT) { $REPORT .= "\n" }
    $REPORT .= $text;
  }

} # append2report


# -------------------------------------------------------------------
sub append2inventory {
  # append2inventory(<key>, <value1>, <value2>, ...)

  # The resulting line is:
  # <key>;<value1>;<value2>;...

  my $Line = uc(shift);

  # while ( my $t = shift ) {
  #   $Line .= ";" . $t
  # }

  while (@_) {
    $Line .= ";" . shift;
  }
  # print "$Line\n";

  if ($INVENTORY) { $INVENTORY .= "\n" }
  $INVENTORY .= $Line;

} # append2inventory






# ===================================================================

# -------------------------------------------------------------------
sub oracle_available {
  title(sub_name());

  # ----- Check if Oracle is available
  my $sql = "select 'database status', status from v\$instance;";

  if (! do_sql($sql)) {return(0)}

  my $V = trim(get_value($TMPOUT1, $DELIM, "database status"));
  print "OPEN=$V\n";

  if ($V ne "OPEN") {
    output_error_message(sub_name() . ": Error: the database status is not 'OPEN'!");
    return(0);
  }
  return(1);

} # oracle_available



# -------------------------------------------------------------------
sub option_usage {

  # 11.2.0.2.0
  # This is currently only matching Oracle version 11.2
  # Use the features instead for earlier Oracle versions.

  if ( $ORACLE_VERSION !~ /^11\.2/ ) { return 0 }

  title(sub_name());

  # -----
  # installed options
  #
  # see
  # Oracle Database Licensing Information
  # 11g Release 2 (11.2)
  # Part Number E10594-26
  # http://docs.oracle.com/cd/E11882_01/license.112/e10594/options.htm
  #
  # The query is based on the option_usage.sql script which can
  # be downloaded at My Oracle Support, Document ID 1317265.1
  #
  # It is a bit lengthy...
  #
  # NOTE: Spatial cannot be checked correctly because the usage is always
  #       incremented, even for the Locator which is free to use.
  #
  # The result looks like:
  #
  # Option/Management Pack                             !Use!ReportGen Time
  # ---------------------------------------------------!---!-------------------
  # Tuning  Pack                                       !YES!2013-01-25 08:31:22
  # Active Data Guard                                  !NO !2013-01-25 08:31:22
  # Advanced Compression                               !NO !2013-01-25 08:31:22
  # Advanced Security                                  !NO !2013-01-25 08:31:22
  # Change Management Pack                             !NO !2013-01-25 08:31:22
  # Configuration Management Pack for Oracle Database  !NO !2013-01-25 08:31:22
  # Data Masking Pack                                  !NO !2013-01-25 08:31:22
  # Data Mining                                        !NO !2013-01-25 08:31:22
  # Database Vault                                     !NO !2013-01-25 08:31:22
  # Diagnostic Pack                                    !NO !2013-01-25 08:31:22
  # Exadata                                            !NO !2013-01-25 08:31:22
  # Label Security                                     !NO !2013-01-25 08:31:22
  # OLAP                                               !NO !2013-01-25 08:31:22
  # Partitioning                                       !NO !2013-01-25 08:31:22
  # Provisioning and Patch Automation Pack             !NO !2013-01-25 08:31:22
  # Provisioning and Patch Automation Pack for Database!NO !2013-01-25 08:31:22
  # Real Application Clusters                          !NO !2013-01-25 08:31:22
  # Real Application Testing                           !NO !2013-01-25 08:31:22
  # Spatial                                            !NO !2013-01-25 08:31:22
  # Total Recall                                       !NO !2013-01-25 08:31:22
  # WebLogic Server Management Pack Enterprise Edition !NO !2013-01-25 08:31:22
  #
  # NOTE: Only the used options will make it to the inventory!


  my $sql = "
    with features as(
      select a OPTIONS, b NAME  from (
        select 'Active Data Guard' a,  'Active Data Guard - Real-Time Query on Physical Standby' b from dual
        union all
        select 'Advanced Compression', 'HeapCompression' from dual
        union all
        select 'Advanced Compression', 'Backup BZIP2 Compression' from dual
        union all
        select 'Advanced Compression', 'Backup DEFAULT Compression' from dual
        union all
        select 'Advanced Compression', 'Backup HIGH Compression' from dual
        union all
        select 'Advanced Compression', 'Backup LOW Compression' from dual
        union all
        select 'Advanced Compression', 'Backup MEDIUM Compression' from dual
        union all
        select 'Advanced Compression', 'Backup ZLIB, Compression' from dual
        union all
        select 'Advanced Compression', 'SecureFile Compression (user)' from dual
        union all
        select 'Advanced Compression', 'SecureFile Deduplication (user)' from dual
        union all
        select 'Advanced Compression', 'Data Guard' from dual
        union all
        select 'Advanced Compression', 'Oracle Utility Datapump (Export)' from dual
        union all
        select 'Advanced Compression', 'Oracle Utility Datapump (Import)' from dual
        union all
        select 'Advanced Security',    'ASO native encryption and checksumming' from dual
        union all
        select 'Advanced Security',    'Transparent Data Encryption' from dual
        union all
        select 'Advanced Security',    'Encrypted Tablespaces' from dual
        union all
        select 'Advanced Security',    'Backup Encryption' from dual
        union all
        select 'Advanced Security',    'SecureFile Encryption (user)' from dual
        union all
        select 'Change Management Pack', 'Change Management Pack (GC)' from dual
        union all
        select 'Data Masking Pack',     'Data Masking Pack (GC)' from dual
        union all
        select 'Data Mining',           'Data Mining' from dual
        union all
        select 'Diagnostic Pack',       'Diagnostic Pack' from dual
        union all
        select 'Diagnostic Pack',       'ADDM' from dual
        union all
        select 'Diagnostic Pack',       'AWR Baseline' from dual
        union all
        select 'Diagnostic Pack',       'AWR Baseline Template' from dual
        union all
        select 'Diagnostic Pack',       'AWR Report' from dual
        union all
        select 'Diagnostic Pack',       'Baseline Adaptive Thresholds' from dual
        union all
        select 'Diagnostic Pack',       'Baseline Static Computations' from dual
        union all
        select 'Tuning  Pack',          'Tuning Pack' from dual
        union all
        select 'Tuning  Pack',          'Real-Time SQL Monitoring' from dual
        union all
        select 'Tuning  Pack',          'SQL Tuning Advisor' from dual
        union all
        select 'Tuning  Pack',          'SQL Access Advisor' from dual
        union all
        select 'Tuning  Pack',          'SQL Profile' from dual
        union all
        select 'Tuning  Pack',          'Automatic SQL Tuning Advisor' from dual
        union all
        select 'Database Vault',        'Oracle Database Vault' from dual
        union all
        select 'WebLogic Server Management Pack Enterprise Edition',    'EM AS Provisioning and Patch Automation (GC)' from dual
        union all
        select 'Configuration Management Pack for Oracle Database',     'EM Config Management Pack (GC)' from dual
        union all
        select 'Provisioning and Patch Automation Pack for Database',   'EM Database Provisioning and Patch Automation (GC)' from dual
        union all
        select 'Provisioning and Patch Automation Pack',        'EM Standalone Provisioning and Patch Automation Pack (GC)' from dual
        union all
        select 'Exadata',               'Exadata' from dual
        union all
        select 'Label Security',        'Label Security' from dual
        union all
        select 'OLAP',                  'OLAP - Analytic Workspaces' from dual
        union all
        select 'Partitioning',          'Partitioning (user)' from dual
        union all
        select 'Real Application Clusters',  'Real Application Clusters (RAC)' from dual
        union all
        select 'Real Application Testing',   'Database Replay: Workload Capture' from dual
        union all
        select 'Real Application Testing',   'Database Replay: Workload Replay' from dual
        union all
        select 'Real Application Testing',   'SQL Performance Analyzer' from dual
        union all
        select 'Spatial',        'Spatial (Not used because this does not differential usage of spatial over locator, which is free)' from dual
        union all
        select 'Total Recall',   'Flashback Data Archive' from dual
      )  -- select
    )    -- with
    select
         t.o \"Option/Management Pack\"
       , t.u \"Used\"
       -- ,d.DBID   \"DBID\"
       -- ,d.name   \"DB Name\"
       -- ,i.version  \"DB Version\"
       -- ,i.host_name  \"Host Name\"
       -- ,to_char(sysdate, 'YYYY-MM-DD HH24:MI:SS') \"ReportGen Time\"
    from (
      select OPTIONS o, DECODE(sum(num),0,'NO','YES') u
      from (
        select f.OPTIONS OPTIONS, case
          when f_stat.name is null then 0
          when (
            (f_stat.currently_used = 'TRUE' and
             f_stat.detected_usages > 0 and
             (sysdate - f_stat.last_usage_date) < 366 and
             f_stat.total_samples > 0)
            or
             (f_stat.detected_usages > 0 and (sysdate - f_stat.last_usage_date) < 366 and f_stat.total_samples > 0)
          ) and (
            f_stat.name not in('Data Guard', 'Oracle Utility Datapump (Export)', 'Oracle Utility Datapump (Import)')
            or
            (f_stat.name in('Data Guard', 'Oracle Utility Datapump (Export)', 'Oracle Utility Datapump (Import)') and
             f_stat.feature_info is not null and trim(substr(to_char(feature_info), instr(to_char(feature_info), 'compression used: ',1,1) + 18, 2)) != '0')
          )
          then 1
          else 0
         end num
        from features f, sys.dba_feature_usage_statistics f_stat
        where f.name = f_stat.name(+)
      ) group by options
    ) t, v\$instance i, v\$database d
    order by 2 desc,1
    ;
  ";

  if (! do_sql($sql)) {return(0)}

  my @O = ();
  get_value_lines(\@O, $TMPOUT1);

  # - options
  foreach my $i (@O) {
    my @E = split($DELIM, $i);
    @E = map(trim($_), @E);

    if ( uc($E[1]) eq "YES" ) {
      # Only USED options are reported in the inventory
      append2inventory("ORACLEOPTIONUSAGE", $E[0]);
    }
  }

  # - report

  # prepend a title line
  unshift(@O, "Option/Management Pack  $DELIM  Use ");

  my $txt = make_text_report(\@O, $DELIM, "LL", 1);

  append2report("=", "option usage", $txt);

  return(1);

} # option_usage



# -------------------------------------------------------------------
sub part_1 {
  # Oracle version, components (what is installed), features, options, opatch lsinventory, environment variables

  title(sub_name());

  my $dt = iso_datetime();
  append2report("*", "Oracle Database Information Report", "Generated: $dt");

  my $sql = "
    select 'oracle version', version from v\$instance;
    select 'hostname', host_name from v\$instance;
    select 'platform', platform_name from v\$database;
    select 'instance name', instance_name from v\$instance;
    select 'instance startup at', TO_CHAR(startup_time,'YYYY-MM-DD HH24:MI:SS') from v\$instance;
    select 'DBID', dbid from v\$database;
  ";

  if (! do_sql($sql)) {return(0)}

  $ORACLE_VERSION     = trim(get_value($TMPOUT1, $DELIM, "oracle version"));
  # e.g. 10.1.0.3.0, 10.2.0.3.0

  append2report("=", "General Info", "");

  append2report("", "", "Oracle Version     : $ORACLE_VERSION");
  append2inventory("# date generated", $dt);
  append2inventory("ORACLEVERSION", $ORACLE_VERSION);

  my $my_instance = trim(get_value($TMPOUT1, $DELIM, "instance name"));
  append2report("", "", "Instance name      : $my_instance");
  append2inventory("ORACLEINSTANCE", $my_instance);

  append2report("", "", "Instance startup at: " . trim(get_value($TMPOUT1, $DELIM, "instance startup at")));
  # DBID is already in report in "database settings".
  # append2report("", "", "DBID               : " . trim(get_value($TMPOUT1, $DELIM, "DBID")));

  $MY_HOST = trim(get_value($TMPOUT1, $DELIM, "hostname"));
  append2report("", "", "Running on host    : $MY_HOST");
  append2inventory("ORACLEHOST", $MY_HOST);

  append2report("", "", "Host platform      : " . trim(get_value($TMPOUT1, $DELIM, "platform")));

  # Save for later usage.
  my $dbid = trim(get_value($TMPOUT1, $DELIM, "DBID"));

  # -----
  # Environment variables

  append2report("=", "Environment Variables", "");
  append2report("", "", "ORACLE_BASE: $ENV{'ORACLE_BASE'}");
  append2report("", "", "ORACLE_HOME: $ENV{'ORACLE_HOME'}");
  append2report("", "", "ORACLE_SID : $ENV{'ORACLE_SID'}");
  append2report("", "", "TNS_ADMIN  : $ENV{'TNS_ADMIN'}");
  append2report("", "", "NLS_LANG   : $ENV{'NLS_LANG'} (*)");
  append2report("", "", "(*) NLS_LANG may not reflect the database settings! Just the environment variable!");


  # -----
  # banner information
  # standard edition/enterprise edition

  $sql = "select banner from v\$version; ";

  if (! do_sql($sql)) {return(0)}

  my @L = ();
  my $txt = "";

  if (get_value_lines(\@L, $TMPOUT1)) {
    $txt = make_text_report(\@L, $DELIM, "L", 0);
    append2report("=", "banner", $txt);
  }


  # -----
  # components

  $sql = "
    select
      COMP_NAME, VERSION, STATUS,
      to_char(to_date(MODIFIED, 'DD-MON-YYYY HH24:MI:SS'), 'YYYY-MM-DD HH24:MI:SS')
    from dba_registry
    order by COMP_NAME
    ;
  ";

  if (! do_sql($sql)) {return(0)}

  @L = ();
  get_value_lines(\@L, $TMPOUT1);

  # - inventory
  foreach my $i (@L) {
    my @E = split($DELIM, $i);
    @E = map(trim($_), @E);

    append2inventory("ORACLECOMPONENT", $E[0], $E[1]);
  }

  # - report

  # prepend a title line
  unshift(@L, "component  $DELIM  version  $DELIM  status  $DELIM  last modified");

  $txt = make_text_report(\@L, $DELIM, "", 1);

  append2report("=", "components", $txt);


  # -----
  # installed features
  # 
  # NOTE: That are NOT the options!
  #
  # see
  # Oracle Database Licensing Information
  # 11g Release 2 (11.2)
  # Part Number E10594-26
  # http://docs.oracle.com/cd/E11882_01/license.112/e10594/options.htm

  $sql = "
    select
      NAME, VERSION, DETECTED_USAGES,
      nvl(to_char(FIRST_USAGE_DATE, 'YYYY-MM-DD HH24:MI:SS'), '-'),
      nvl(to_char(LAST_USAGE_DATE, 'YYYY-MM-DD HH24:MI:SS'), '-')
    from dba_feature_usage_statistics
    where DBID = $dbid
    order by NAME, VERSION
    ;
  ";

  if (! do_sql($sql)) {return(0)}

  @L = ();
  get_value_lines(\@L, $TMPOUT1);

  if ( uc($CFG{"ORA_DBINFO.ORACLE_FEATURES"}) eq 'YES' ) {
    # Only if the Oracle features are actively set as option

    # - inventory
    foreach my $i (@L) {
      my @E = split($DELIM, $i);
      @E = map(trim($_), @E);

      append2inventory("ORACLEFEATURE", $E[0], $E[1], $E[2]);
    }
  }

  # - report

  # prepend a title line
  unshift(@L, "feature  $DELIM  version  $DELIM  usages  $DELIM  first usage $DELIM last usage");

  # note: that one throws many errors, because the dates are missing in many lines!
  $txt = make_text_report(\@L, $DELIM, "LLRLL", 1);

  append2report("=", "installed feature", $txt);

  # -----
  option_usage();


  # -----
  # opatch lsinventory

  # Find the opatch routine
  my $opatch = $ENV{"ORACLE_HOME"} . "/OPatch/opatch";

  my $cmd = "$opatch lsinventory";

  $txt = `$cmd`;

  append2report("=", "opatch lsinventory", $txt);

  # -----
  # Find all installed patches from output of opatch lsinventory

  my @lines = split /\n/, $txt;
  foreach my $line (@lines) {

    # Patch  6494745      : applied on Mon Jul 06 13:37:48 CEST 2009
    if ( $line =~ /^Patch  \d+/i ) {

      my ($patch, $patchno, $dummy) = split(/\s+/, $line);
      print "patch=$patchno\n";

      append2inventory("ORACLEPATCH", $patchno);
    }
  } # foreach

  return(1);
} # part_1


# -------------------------------------------------------------------
sub part_2 {
  # Parameters,  NLS settings

  title(sub_name());

  my @L = ();
  my $txt = "";

  # -----
  # Parameters

  my $sql = "
    select rtrim(name)
    , rtrim(value)
    , rtrim(isdefault)
    from v\$parameter
    order by name;
  ";

  if (! do_sql($sql)) {return(0)}

  @L = ();
  get_value_lines(\@L, $TMPOUT1);

  unshift(@L, "parameter  $DELIM  value $DELIM is default");

  $txt = make_text_report(\@L, $DELIM, "LLL", 1);

  append2report("=", "parameters", $txt);

  # -----
  # NLS database settings

  $sql = "
    select parameter, value
    from nls_database_parameters
    order by parameter;
  ";

  if (! do_sql($sql)) {return(0)}

  @L = ();
  get_value_lines(\@L, $TMPOUT1);

  # Insert title
  unshift(@L, "parameter  $DELIM  value");

  $txt = make_text_report(\@L, $DELIM, "", 1);

  append2report("=", "NLS database parameters", $txt);

  # -----
  # NLS instance settings

  $sql = "
    select parameter, value
    from nls_instance_parameters
    order by parameter;
  ";

  if (! do_sql($sql)) {return(0)}

  @L = ();
  get_value_lines(\@L, $TMPOUT1);

  # Insert title
  unshift(@L, "parameter  $DELIM  value");

  $txt = make_text_report(\@L, $DELIM, "", 1);

  append2report("=", "NLS instance parameters", $txt);


} # part_2


# -------------------------------------------------------------------
sub part_3 {
  # Database id, archive logging, flashback

  title(sub_name());

  my @L = ();
  my $txt = "";

  my $sql = "
    select 'DBID',           DBID from v\$database;
    select 'NAME',           NAME from v\$database;
    select 'CREATED',        to_char(CREATED, 'YYYY-MM-DD HH24:MI:SS') from v\$database;
    select 'DATABASE_ROLE',  DATABASE_ROLE from v\$database;
    select 'GUARD_STATUS',   GUARD_STATUS from v\$database;
    select 'DB_UNIQUE_NAME', DB_UNIQUE_NAME from v\$database;
    select 'FLASHBACK_ON',   FLASHBACK_ON from v\$database;
    select 'LOG_MODE',       LOG_MODE from v\$database;
  ";

  if (! do_sql($sql)) {return(0)}

  append2report("=", "Database settings");
  append2report("", "", "Database ID          : " . trim(get_value($TMPOUT1, $DELIM, "DBID")));
  append2report("", "", "Database name        : " . trim(get_value($TMPOUT1, $DELIM, "NAME")));
  append2report("", "", "Database unique name : " . trim(get_value($TMPOUT1, $DELIM, "DB_UNIQUE_NAME")));
  append2report("", "", "Database created     : " . trim(get_value($TMPOUT1, $DELIM, "CREATED")));
  append2report("", "", "Database role        : " . trim(get_value($TMPOUT1, $DELIM, "DATABASE_ROLE")));
  append2report("", "", "Archive log mode     : " . trim(get_value($TMPOUT1, $DELIM, "LOG_MODE")));
  append2report("", "", "Flashback            : " . trim(get_value($TMPOUT1, $DELIM, "FLASHBACK_ON")));
  append2report("", "", "Dataguard status     : " . trim(get_value($TMPOUT1, $DELIM, "GUARD_STATUS")));

} # part_3


# -------------------------------------------------------------------
sub part_4 {
  # RAC

  title(sub_name());

  append2report("=", "RAC", "Information about RAC environments is currently not implemented.");

} # part_4


# -------------------------------------------------------------------
sub buffer_cache9 {

  append2report("=", "Buffer Caches", "This is currently not supported for Oracle 9.");

} # buffer_cache9


# -------------------------------------------------------------------
sub part_5 {
  # SGA, shared pool, buffer caches, pga

  title(sub_name());

  my $txt = "";
  my @L = ();

  # -----
  # SGA

  my $sql = "
    select 'overall size', sum(value) from v\$sga;
    select 'free memory', current_size from v\$sga_dynamic_free_memory;
  ";

  if (! do_sql($sql)) {return(0)}

  my $size = trim(get_value($TMPOUT1, $DELIM, "overall size"));
  $size = sprintf("%10.1f MB", $size/(1024*1024));

  my $free = trim(get_value($TMPOUT1, $DELIM, "free memory"));
  $free = sprintf("%10.1f MB", $free/(1024*1024));

  append2report("=", "SGA");
  append2report("", "", "Size : $size");
  append2report("", "", "Free : $free");

  # -----
  # Buffer Caches

  # for Oracle 9.2
  if ($ORACLE_VERSION =~ /^9/) {
    # Use the old style for Oracle 9
    buffer_cache9();
    return();
  }

  # Here only for Oracle 10, 11

  # current_size is in MB

  $sql = "
    select
        name || ' (' || block_size / 1024 || 'k)'
      , current_size
    from v\$buffer_pool
    order by name, block_size
    ;
  ";

  if (! do_sql($sql)) {return(0)}

  @L = ();
  get_value_lines(\@L, $TMPOUT1);

  unshift(@L, "buffer cache  $DELIM  size [MB]");

  $txt = make_text_report(\@L, $DELIM, "LR", 1);

  append2report("-", "Buffer Caches", $txt);

  # -----
  # shared pool

  $sql = "
    select 'shared pool size', current_size from v\$sga_dynamic_components
      where component = 'shared pool';
  ";

  if (! do_sql($sql)) {return(0)}

  my $V = int(trim(get_value($TMPOUT1, $DELIM, 'shared pool size')));
  $V = sprintf("%10.1f MB", $V/(1024*1024));

  append2report("-", "Shared Pool", "Size : $V");


  # -----
  # PGA

  $sql = "select name, value from v\$pgastat;";

  if (! do_sql($sql)) {return(0)}

  append2report("-", "PGA");

  $V = trim(get_value($TMPOUT1, $DELIM, "aggregate PGA target parameter"));
  $V = sprintf("%10.1f MB", $V/(1024*1024));
  append2report("", "", "aggregate PGA target parameter : $V");

  $V = trim(get_value($TMPOUT1, $DELIM, "aggregate PGA auto target"));
  $V = sprintf("%10.1f MB", $V/(1024*1024));
  append2report("", "", "aggregate PGA auto target      : $V");

  $V = trim(get_value($TMPOUT1, $DELIM, "maximum PGA allocated"));
  $V = sprintf("%10.1f MB", $V/(1024*1024));
  append2report("", "", "maximum PGA allocated          : $V");

} # part_5


# -------------------------------------------------------------------
sub part_6 {
  # Online Redo Logs

  title(sub_name());

  my $txt = "";
  my @L = ();

  # -----
  # Online Redo Logs

  my $sql = "
    select 
      l1.group#
      , member
      , l1.status
      , type
      , bytes
    from v\$log l1, v\$logfile l2
    where l1.GROUP# = l2.GROUP#
    order by l1.GROUP#
    ;
  ";

  if (! do_sql($sql)) {return(0)}

  @L = ();
  get_value_lines(\@L, $TMPOUT1);

  # Insert title
  unshift(@L, "group $DELIM member $DELIM status $DELIM type $DELIM size");

  $txt = make_text_report(\@L, $DELIM, "RLLLR", 1);

  append2report("=", "Online Redo Logs", $txt);

} # part_6


# -------------------------------------------------------------------
sub part_7 {

  # ASM

  title(sub_name());



} # part_7



# -------------------------------------------------------------------
sub datafiles {
  # datafiles(<tablespace>, <contents>);

  my ($ts, $contents) = @_;

  my $sql;

  if ($contents eq "TEMPORARY") {

    $sql = "
      select
          file_name
        , round(bytes / (1024 * 1024))
        , round(maxbytes / (1024 * 1024))
        , status
        , autoextensible
      from dba_temp_files
      where upper(tablespace_name) = upper('$ts')
      order by file_id;
    ";

  } else {

    $sql = "
      select
          file_name
        , round(bytes / (1024 * 1024))
        , round(maxbytes / (1024 * 1024))
        , status
        , autoextensible
      from dba_data_files
      where upper(tablespace_name) = upper('$ts')
      order by file_id;
    ";

  }

  if (! do_sql($sql)) {return(0)}

  my @DF;
  get_value_lines(\@DF, $TMPOUT1);

  # prepend a title line
  unshift(@DF, "data file $DELIM size [MB] $DELIM max size [MB] $DELIM status $DELIM autoextensible");

  my $txt = make_text_report(\@DF, $DELIM, "LRRLL", 1);

  append2report("", "", $txt);

} # datafiles

# -------------------------------------------------------------------
sub part_8 {
  # part_8();
  # 
  # Tablespaces, datafiles

  title(sub_name());

  # Not sure, if this will work for older versions.
  my $sql = "
    select tablespace_name, initial_extent, next_extent,
           status, contents, logging, extent_management,
           allocation_type, segment_space_management,
           (select count(*) from dba_data_files ddf
              where autoextensible = 'YES'
                and dts.tablespace_name = ddf.tablespace_name) +
           (select count(*) from dba_temp_files dtf
              where autoextensible = 'YES'
                and dts.tablespace_name = dtf.tablespace_name),
           (select count(*) from dba_data_files ddf
              where dts.tablespace_name = ddf.tablespace_name) +
           (select count(*) from dba_temp_files dtf
              where dts.tablespace_name = dtf.tablespace_name),
           block_size, 
           PCT_INCREASE,
           encrypted,
           DEF_TAB_COMPRESSION, COMPRESS_FOR
      from dba_tablespaces dts order by 1;
  ";

  if (! do_sql($sql)) {return(0)}

  append2report("=", "Tablespaces", "");

  my @TSPACE;
  get_value_lines(\@TSPACE, $TMPOUT1);

  foreach my $tspace (@TSPACE) {
    my @E = split($DELIM, $tspace);
    @E = map(trim($_), @E);

    # -----
    # tablespace infos
    append2report("~", $E[0]);
    append2report("", "", "Contents                 : " . $E[4]);
    append2report("", "", "Block Size               : " . $E[11]);
    append2report("", "", "Status                   : " . $E[3]);
    append2report("", "", "Encrypted                : " . $E[13]);
    append2report("", "", "Logging                  : " . $E[5]);
    if ($E[14] eq "DISABLED") {
      append2report("", "", "Compression              : " . $E[14]);
    } else {
      append2report("", "", "Compression              : " . $E[14] . " for " . $E[15]);
    }
    append2report("", "", "Extent Management        : " . $E[6]);
    append2report("", "", "Initial Extent           : " . $E[1]);
    append2report("", "", "Percent Increase         : " . $E[12]);
    append2report("", "", "Next Extent              : " . $E[2]);
    append2report("", "", "Allocation Type          : " . $E[7]);
    append2report("", "", "Segment Space Management : " . $E[8]);
    append2report("", "", "Datafiles                : " . $E[10]);
    append2report("", "", "Autoextensible datafiles : " . $E[9]);
    append2report("", "", "");

    # tablespace name, contents
    datafiles($E[0], $E[4]);

    append2report("", "", "");

  } # foreach

} # part_8


# -------------------------------------------------------------------
sub part_9 {
  # fast recovery area
  # Not available before Oracle 10
  if ($ORACLE_VERSION !~ /^1\d/) { return(1) }

  title(sub_name());

  my $sql = "";

  # -----
  # Check if fast recovery area is in use

  $sql = "
    select upper(name), value from v\$parameter
      where upper(name)
        in ('DB_RECOVERY_FILE_DEST_SIZE', 'DB_RECOVERY_FILE_DEST');
  ";

  if (! do_sql($sql)) {return(0)}

  # 'db_recovery_file_dest' could be used for RMAN backups,
  # even if flashback feature is not enabled.

  my $db_recovery_file_dest = trim(get_value($TMPOUT1, $DELIM, 'DB_RECOVERY_FILE_DEST'));
  my $db_recovery_file_dest_size = trim(get_value($TMPOUT1, $DELIM, 'DB_RECOVERY_FILE_DEST_SIZE'));

  if ($db_recovery_file_dest && $db_recovery_file_dest_size) {
    # FRA is used

    $sql = "
      select 'fra_usage'
      , name , space_limit , space_used
      from v\$recovery_file_dest;
    ";
    # this will currently return just one record

    if (! do_sql($sql)) {return(0)}

    my $V = trim(get_value($TMPOUT1, $DELIM, 'fra_usage', 2));
    append2report("=", "Fast Recovery Area", "Destination : $V");

    $V = trim(get_value($TMPOUT1, $DELIM, 'fra_usage', 3));
    $V = sprintf("%10.1f MB", $V / (1024*1024));
    append2report("", "", "Size        : $V");

    $V = trim(get_value($TMPOUT1, $DELIM, 'fra_usage', 4));
    $V = sprintf("%10.1f MB", $V / (1024*1024));
    append2report("", "", "Used        : $V");

  } else {
    append2report("=", "Fast Recovery Area", "Not used.");
  }

} # part_9


# -------------------------------------------------------------------
sub part_10 {
  # pfile/spfile, control.txt

  title(sub_name());

  # -----
  # pfile/spfile

  # create pfile=/path/to/backup.ora from spfile;

  my $spfilename = "";
  my $spfile_is_temporary = 0;
  my $pfilename  = "";
  # temporary directory to place the copy of the [s]pfile
  my $tempdir  = $ENV{"TMP"} || $ENV{"TEMP"} || "/tmp";
  my $filenameext = "_for_$0_only.ora";

  # -----
  # Get the [s]pfile settings
  # BUT: PFILE does not need to be set!!!

  my $sql = "
    select 
      upper(name), rtrim(value)
    from v\$parameter 
    where upper(name) in ('PFILE', 'SPFILE');
  ";

  if (! do_sql($sql)) {return(0)}

  $spfilename = trim(get_value($TMPOUT1, $DELIM, "SPFILE"));
  $pfilename  = trim(get_value($TMPOUT1, $DELIM, "PFILE"));
  print "PFILE: $pfilename, SPFILE: $spfilename\n";

  if (! $spfilename) {
    print "SPFILE parameter is not set => build a temporary spfile.\n";

    # Use a none default filename! Oracle would use it otherwise at next startup.
    $spfilename = "$tempdir/spfile_" . $ENV{"ORACLE_SID"} . $filenameext;

    $sql = "create spfile='$spfilename' from pfile;";

    if (! do_sql($sql)) {return(0)}

    print "Temporary spfile '$spfilename' created from pfile.\n";
    $spfile_is_temporary = 1;

  } else {

    print "SPFILE parameter is set: $spfilename\n";

  }
  # Now, we do have an spfile (whether by default or explicitly created)
  # and can create a readable parameter file (init.ora) out of it.

  # -----
  $pfilename = "$tempdir/init_" . $ENV{"ORACLE_SID"} . $filenameext;

  $sql = "create pfile = '$pfilename' from spfile = '$spfilename';";

  if (! do_sql($sql)) {return(0)}

  append_file_contents($pfilename, "spfile");

  unlink($pfilename);
  if ($spfile_is_temporary) { unlink($spfilename) }

  # -----
  # control.txt

  my $cfile = "$tempdir/control.txt";

  $sql = "
    ALTER DATABASE BACKUP CONTROLFILE TO TRACE AS '$cfile' REUSE;
  ";

  if (! do_sql($sql)) {return(0)}

  append_file_contents($cfile, "controlfile");

  unlink($cfile);

} # part_10



# -------------------------------------------------------------------
sub read_file {
  # read_file(<filename>);

  my ($filename) = @_;

  my $txt = "";

  if (! open(INFILE, "<", $filename)) {
    output_error_message(sub_name() . ": Error: Cannot open '$filename' for reading. $!");
    return(undef);
  }
  while (my $L = <INFILE>) {
    chomp($L);
    if ($txt) { $txt .= "\n" }
    $txt .= $L;
  } # while

  if (! close(INFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$filename'. $!");
    return(undef);
  }

  return($txt);

} # read_file


# -------------------------------------------------------------------
sub append_file_contents {
  my ($filename, $title) = @_;

  # Default title is the basename of the file
  if (! $title) { $title = basename($filename) }

  my $txt = "File '$filename' could not be found or is not readable!";

  if (-r $filename) {
    # Read in the contents of the file
    $txt = read_file($filename);

    if ($txt) { 
      $txt = "-----\n" . $txt . "\n-----";
    } else {
      $txt = "File '$filename' has no contents!";
    }
  }

  append2report("=", $title, "Filename: $filename");
  append2report("", "", $txt);

} # append_file_contents


# -------------------------------------------------------------------
sub part_11 {
  # listener.ora, tnsnames.ora, sqlnet.ora

  title(sub_name());

  my $txt = "";

  my $nw_admin = $ENV{"ORACLE_HOME"} . "/network/admin";
  if ($ENV{"TNS_ADMIN"} ) {
    $nw_admin = $ENV{"TNS_ADMIN"};
  }

  # -----
  # sqlnet.ora

  my $filename = $nw_admin . "/sqlnet.ora";
  append_file_contents($filename);

  # -----
  # listener.ora

  $filename = $nw_admin . "/listener.ora";
  append_file_contents($filename);

  # -----
  # tnsnames.ora

  $filename = $nw_admin . "/tnsnames.ora";
  append_file_contents($filename);

} # part_11



# -------------------------------------------------------------------
sub bzip2_file {
  my ($filename) = @_;

  if (-r $filename) {
    my $cmd = "bzip2 $filename";
    system($cmd);

    if ($? == -1) {
      # bzip2 may not be available
      print STDERR "$CURRPROG: Error: failed to execute command '$cmd': $!\n";
      return(undef);
    }
    elsif ($? & 127) {
      printf "$CURRPROG: Error: child died with signal %d, %s coredump\n", ($? & 127), ($? & 128) ? 'with' : 'without';
      return(undef);
    }
    else {
      print "Command '$cmd' exited with value ", $? >> 8, "\n";
      $filename .= ".bz2";
    }
  }
  return($filename);
} # bzip2_file


# -------------------------------------------------------------------
sub send2uls {
  # send2uls(<filename>);
  #
  # Write the report to a file and send that to ULS.

  title(sub_name());

  my ($filename) = @_;
  my $uls_detail = "Oracle Database Information Report";
  my $alias_filename = "ora_dbinfo.txt";

  if (! open(OUTFILE, ">", $filename)) {
    output_error_message(sub_name() . ": Error: Cannot open '$filename' for writing. $!");
    return(0);
  }

  print OUTFILE $REPORT, "\n";

  if (! close(OUTFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$filename'. $!");
    return(0);
  }

  # bzip2 the file, if possible
  if ( $CFG{"ORA_DBINFO.BZIP_THE_REPORT"} ) {
    my $new_filename = bzip2_file($filename);
    if ($new_filename) { 
      # bzip2 has worked!
      $filename = $new_filename;
      $alias_filename .= ".bz2";
    }
  }

  if ($CFG{"ORA_DBINFO.AS_SERVER_DOC"} ) {
    # Send the report as server documentation
    uls_server_doc({
        name        => $IDENTIFIER
      , description => $uls_detail
      , filename    => $filename
      , rename_to   => $alias_filename
    });
  }

  if ($CFG{"ORA_DBINFO.AS_VALUE"} ) {
    # Send the report in the Oracle DB [<sid>] section

    uls_file({
        teststep  => $IDENTIFIER
      , detail    => $uls_detail
      , filename  => $filename
      , rename_to => $alias_filename
   });
  }

} # send2uls



# -------------------------------------------------------------------
sub sendinventory2uls {
  # sendinventory2uls(<filename>);
  #
  # Write the report to a file and send that to ULS.

  title(sub_name());

  my ($filename) = @_;

  my $description = "# Inventory of an Oracle Database:
# Oracle software version
# ORACLEVERSION;<version>
# ORACLEVERSION;11.2.0.2.0
# Name of the Oracle database instance
# ORACLEINSTANCE;<orcl>
# Hostname on which the Oracle database runs
# ORACLEHOST;<hostname>
# List of all installed Oracle components
# ORACLECOMPONENT;<component>;<version>
# ORACLECOMPONENT;Oracle Expression Filter;11.2.0.2.0
# List of all installed Oracle features
# ORACLEFEATURE;<feature>;<version>;<number of times used> 
# ORACLEFEATURE;AWR Baseline;11.2.0.2.0;0
# List of all used, extra chargeable options, based on the option_usage.sql, ID 1317265.1
# ORACLEOPTIONUSAGE;<option>
# ORACLEOPTIONUSAGE;Provisioning and Patch Automation Pack for Database
# List of all installed patches as found by 'opatch lsinventory'
# ORACLEPATCH;<patch_number>
# ORACLEPATCH;7154843
";

  print "----- inventory -----\n";
  print "$description\n";
  print "$INVENTORY\n";
  print "---------------------\n";

  if (! open(OUTFILE, ">", $filename)) {
    output_error_message(sub_name() . ": Error: Cannot open '$filename' for writing. $!");
    return(0);
  }

  print OUTFILE $description, "\n";
  print OUTFILE $INVENTORY, "\n";

  if (! close(OUTFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$filename'. $!");
    return(0);
  }

  # Send the inventory as server documentation
  uls_server_doc({
      hostname    => $MY_HOST
    , name        => "Inventory-Oracle"
    , description => $description
    , filename    => $filename
    , rename_to   => "Inventory-Oracle-$MY_HOST.csv"
  });

} # sendinventory2uls



# -------------------------------------------------------------------


# ===================================================================
# main
# ===================================================================
#
# initial customization, no output should happen before this.
# The environment must be set up already.

# $CURRPROG = basename($0, ".pl");   # extension is removed
$CURRPROG = basename($0);
my $currdir = dirname($0);
my $start_secs = time;

my $initdir = $ENV{"TMP"} || $ENV{"TEMP"} || $currdir;
my $initial_logfile = "${initdir}/${CURRPROG}_$$.tmp";

# re-direct stdout and stderr to a temporary logfile.
open(STDOUT, "> $initial_logfile") or die "Cannot re-direct STDOUT to $initial_logfile.\n";
open(STDERR, ">&STDOUT") or die "Cannot re-direct STDERR to STDOUT.\n";
select(STDERR);
$| = 1;
select(STDOUT);
$| = 1;           # make unbuffered

# -------------------------------------------------------------------
# From here on, STDOUT+ERR is logged.

title("Start");
print "$CURRPROG is started in directory $currdir\n";

# -------------------------------------------------------------------
# Get configuration file contents

my $cfgfile = $ARGV[0];
print "configuration file=$cfgfile\n";

if (! get_config2($cfgfile, \%CFG, "GENERAL", "ORACLE", "ULS", "ORA_DBINFO")) {
  print STDERR "$CURRPROG: Error: Cannot parse configuration file '$cfgfile' correctly => aborting\n";
  exit(1);
}
show_hash(\%CFG, "=");
print "-----\n\n";

# ----------
# This sets the %ULS to all necessary values
# deriving from %CFG (configuration file),
# environment variables (ULS_*) and defaults.

uls_settings(\%ULS, \%CFG);

show_hash(\%ULS);

# ----------
# Check for IDENTIFIER

# Set default
$IDENTIFIER = $CFG{"ORA_DBINFO.IDENTIFIER"} || "_$CURRPROG";
print "IDENTIFIER=$IDENTIFIER\n";
# From here on, you may use $IDENTIFIER for uniqueness

# -------------------------------------------------------------------
# environment

if (! $ENV{"ORACLE_SID"}) {
  print STDERR "$CURRPROG: Error: ORACLE_SID is not set in the environment => aborting.\n";
  exit(1);
}
if (! $ENV{"ORACLE_HOME"}) {
  print STDERR "$CURRPROG: Error: ORACLE_HOME is not set in the environment => aborting.\n";
  exit(1);
}
print "Oracle environment variables:\n";
print "ORACLE_HOME=", $ENV{"ORACLE_HOME"}, "\n";
print "ORACLE_SID=", $ENV{"ORACLE_SID"}, "\n";
print "\n";


# -------------------------------------------------------------------
# Working directory

my $workdir = $ENV{"WORKING_DIR"} || $CFG{"GENERAL.WORKING_DIR"} || $currdir;

if ( ! (-e $workdir)) {
  print "Creating directory '$workdir' for work files.\n";
  if (! mkdir($workdir)) {
    print STDERR "$CURRPROG: Error: Cannot create directory '$workdir' => aborting!\n";
    exit(1);
  }
}

# prefix for work files
$WORKFILEPREFIX = "${workdir}/${IDENTIFIER}";
# If no oracle sid is found in the workfile prefix, then add it.
# Normally, this should be set in the configuration file.
# if ($WORKFILEPREFIX !~ /$ENV{"ORACLE_SID"}/) { $WORKFILEPREFIX .= "_" . $ENV{"ORACLE_SID"} }
print "WORKFILEPREFIX=$WORKFILEPREFIX\n";

# -------------------------------------------------------------------
# Setting up a lock file to prevent more than one instance of this
# script starting simultaneously.

$LOCKFILE = "${WORKFILEPREFIX}.LOCK";
print "LOCKFILE=$LOCKFILE\n";

if (! lockfile_build($LOCKFILE)) {
  # LOCK file exists and process is still running, abort silently.
  print "Another instance of this script is still running => aborting!\n";
  exit(1);
}

# -------------------------------------------------------------------
# The final log file.

my $logfile = "$WORKFILEPREFIX.log";

move_logfile($logfile);

# re-direct stdout and stderr to a logfile.
open(STDOUT, "> $logfile") or die "Cannot re-direct STDOUT to $logfile. $!\n";
open(STDERR, ">&STDOUT") or die "Cannot re-direct STDERR to STDOUT. $!\n";
select(STDERR);
$| = 1;
select(STDOUT);
$| = 1;           # make unbuffered

# Copy initial logfile contents to current logfile.
if (-e $initial_logfile) {
  print "Contents of initial logfile '$initial_logfile':\n";
  open(INITLOG, $initial_logfile);
  while (<INITLOG>) {print}
  close(INITLOG);
  print "Removing initial log file '$initial_logfile'...";
  if (unlink($initial_logfile)) {print "Done.\n"}
  else {print "Failed.\n"}

  print "Remove possible old temporary files.\n";
  # Remove old .tmp files
  opendir(INITDIR, $initdir);
  my @files = grep(/$CURRPROG.*\.tmp/, map("$initdir/$_", readdir(INITDIR)));
  foreach my $file (@files) {
    # Modification time of file, also fractions of days.
    my $days = pround(-M $file, -1);

    if ($days > 5) {
      print "Remove '", basename($file), "', ($days days old)...";
      if (unlink($file)) {print "Done.\n"}
      else {print "Failed.\n"}
    }
  } # foreach
}

# -------------------------------------------------------------------
title("Set up ULS");

# Initialize uls with basic settings
uls_init(\%ULS);

my $d = iso_datetime($start_secs);
$d =~ s/\d{1}$/0/;

set_uls_timestamp($d);

# uls_show();

# ---- Send name of this script and its version
# uls_value($IDENTIFIER, "script name, version", "$CURRPROG, $VERSION", " ");

uls_value($IDENTIFIER, "script name, version", "$CURRPROG, $VERSION", " ");

uls_timing({
    teststep  => $IDENTIFIER
  , detail    => "start-stop"
  , start     => iso_datetime($start_secs)
});

# -----
# Define some temporary file names
$TMPOUT1 = "${WORKFILEPREFIX}_1.tmp";
print "TMPOUT1=$TMPOUT1\n";
$TMPOUT2 = "${WORKFILEPREFIX}_2.tmp";
print "TMPOUT2=$TMPOUT2\n";

print "DELIM=$DELIM\n";

# -----
# sqlplus command

# Check, if the sqlplus command has been redefined in the configuration file.
$SQLPLUS_COMMAND = $CFG{"ORACLE.SQLPLUS_COMMAND"} || $SQLPLUS_COMMAND;

# -----
# Set the documentation from behind __END__.
# De-reference the return value to the complete hash.
%TESTSTEP_DOC = %{doc2hash(\*DATA)};


# -------------------------------------------------------------------
# The real work starts here.
# ------------------------------------------------------------

if (! oracle_available() ) {
  output_error_message("$CURRPROG: Error: Oracle database is not available => aborting script.");

  clean_up($TMPOUT1, $TMPOUT2, $LOCKFILE);

  send_runtime($start_secs);
  uls_timing($IDENTIFIER, "start-stop", "stop");
  uls_flush(\%ULS);

  exit(1);
}


# -----
# Build everything in hash? array? xml? line-by-line?

# -----
# Oracle version, components (what is installed), options, 
# opatch lsinventory, environment variables

part_1();


# Parameters,  NLS settings
part_2();

# Database id, archive logging, flashback
part_3();


# RAC
part_4();


# SGA, shared pool, buffer caches, pga
part_5();

# Online Redo Logs
part_6();

# ASM
part_7();

# Tablespaces, datafiles
part_8();

# fast recovery area
part_9();


# pfile/spfile, control.txt
part_10();


# listener.ora, tnsnames.ora, sqlnet.ora
part_11();

# -----
# Continue here with more tests.

# The real work ends here.
# -------------------------------------------------------------------

send2uls($TMPOUT1);
# print $REPORT, "\n";

sendinventory2uls($TMPOUT2);
# print "$INVENTORY\n";



# Any errors will have sent already its error messages.
# This is just the final message.
uls_value($IDENTIFIER, "message", $MSG, " ");

send_doc($CURRPROG, $IDENTIFIER);

send_runtime($start_secs);
uls_timing($IDENTIFIER, "start-stop", "stop");

# Do not transfer to ULS
# uls_flush(\%ULS, 1);
#
# Transfer to ULS
uls_flush(\%ULS);

# -------------------------------------------------------------------
clean_up($TMPOUT1, $TMPOUT2, $LOCKFILE);

title("END");

if ($MSG eq "OK") {exit(0)}

exit(1);


#########################
# end of script
#########################

__END__

# The format:
#
# *<teststep title>
# <any text>
# <any text>
#
# *<teststep title>
# <any text>
# <any text>
# ...
#
# Remember to keep the <teststep title> equal to those used
# in the script. If not, you won't get the expected documentation.
#
# Do not use the '\' but use '\\'!

#########################
*ora_dbinfo.pl
==============

This script gathers a lot of information about an Oracle database instance.

This script is part of the ORACLE_TOOLS and works best with the Universal Logging System (ULS). Visit the ULS homepage at http://www.universal-logging-system.org

This script builds a text report about many database characteristics, settings and quantity information. An inventory file in the supported inventory file format is generated. Both are sent to the ULS.

The script is part of the ORACLE_TOOLS package of the Universal Logging System (ULS), see also: http://www.universal-logging-system.org

This script may not cover everything that YOU are interested in, especially no RAC specifics (currently), but it gives quite an overview. Collected information e.g.;
  the name and some settings of the instance, 
  the installed Oracle components,
  a list of used options, 
  a list of installed patches as reported by opatch, 
  a list of all parameters, 
  the NLS settings of the database, 
  the usage of the SGA, buffer caches, shared pool, PGA and tablespaces,
  the usage of the fast recovery area (if used), 
  the contents of the [s]pfile, sqlnet.ora, listener.ora and tnsnames.ora, 
  a textual output of the control file.

You find the inventory information in ULS:
main menu -- administration -- documentation -- server documentation -- <domain> -- <server>

This script is run by a calling script, typically 'ora_dbinfo', that sets the correct environment before starting the Perl script ora_dbinfo.pl. The 'ora_dbinfo' in turn is called by the cron daemon on Un*x or through a scheduled task on Wind*ws. The script generates a log file and several files to keep data for the next run(s). The directory defined by WORKING_DIR in the oracle_tools.conf configuration file is used as the destination for those files.

You may place the scripts in whatever directory you like.

script name, version:
  Sends the name and the current version of this script.

start-stop:
  The start and stop timestamps for the script.

message:
  If the script runs fine, it returns 'OK', else an error message.
  This is intended to monitor the proper execution of this script.

runtime:
  The runtime of the script. This does not include the transmission
  to the ULS.

Oracle Database Information Report:
  A text file (probably bzipped) that contains the report.
  Depending on the user configuration, the report may instead appear 
  at the server documentation section for that server.



Copyright 2011 - 2013, roveda

ORACLE_TOOLS is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

ORACLE_TOOLS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with ORACLE_TOOLS.  If not, see <http://www.gnu.org/licenses/>.

